'''
Interface for general interaction and parser
'''
from abc import ABC, abstractmethod
import csv
from itertools import islice
import os


class InteractionParser(ABC):
    @abstractmethod
    def all_users(self): # the list of all users
        pass

    def num_users(self):
        return len(self.all_users())

    @abstractmethod
    def num_interactions(self, user):
        pass

    @abstractmethod
    def parse_interactions(self, user, start_index, end_index):
        pass


class BasicCSVInteractionParser(InteractionParser, ABC):
    def __init__(self, data_root, has_header):
        # traverse through all files in data_root
        self._all_users = []
        self._data_root = data_root
        self._all_users = os.listdir(data_root)
        # walk through all subdirectories for tree-structured data
        # for dirpath, _, filenames in os.walk(data_root):
        #     self._all_users.extend([os.path.join(dirpath, file) for file in filenames])
        # store number of interactions for later
        self._num_inter_cached = {}
        self._has_header = has_header

    def open_user_file(self, user):
        return open(os.path.join(self._data_root, user), "r")

    def all_users(self):
        return self._all_users

    def num_interactions(self, user):
        if user in self._num_inter_cached:
            return self._num_inter_cached[user]
        else:
            with self.open_user_file(user) as f:
                nlines = len(f.readlines())
            ninters = nlines - 1 if self._has_header else nlines
            self._num_inter_cached[user] = ninters
            return ninters

    def parse_interactions(self, user, start_index, end_index):
        with self.open_user_file(user) as f:
            reader = csv.reader(f)
            # skip header if exists
            if self._has_header:
                next(reader)
            # only look from start_index to end_index
            rows = islice(reader, start_index, end_index)
            return [self.parse_single_interaction(row) for row in rows]

    @abstractmethod
    def parse_single_interaction(self, row):
        pass
